<?php
/**
* A theme's functions.php file is primarily for registering various items
* which the theme supports. This is generally done through functions called
* by hooks (add_action()). Note: if a child theme, both the parent's and child's functions.php files are included automatically by Word Press (child's first).
*/

/**
* Notes on theme URI functions
*
* get_template_directory_uri();
* returns the URL of the parent theme if applicable or active theme's folder if not
* example: http://www.example.com/wp-content/themes/digitalhill
*
* get_stylesheet_directory_uri();
* returns the URL of the active (child) theme's folder
* example: http://www.example.com/wp-content/themes/digitalhill-child
*
* get_stylesheet_uri();
* effectively get_stylesheet_directory_uri() . 'style.css';
* example: http://www.example.com/wp-content/themes/digitalhill-child/style.css
*
* get_theme_file_uri( $file );
* if $file is empty, returns get_stylesheet_directory_uri()
* checks stylesheet (child) folder for file and returns URL if found
* if not found, returns the URL of file in template (parent) folder
* example: http://www.example.com/wp-content/themes/digitalhill/file.php
* It's best to use this function in most cases as it allows a child theme to
* override a parent if needed
*/

/**
* Sets up theme defaults and registers support for various WordPress features.
*
* Note that this function is hooked into the after_setup_theme hook, which
* runs before the init hook. The init hook is too late for some features, such
* as indicating support for post thumbnails.
*/

require_once('includes/class.DevLog.php');

function dhwp_setup() {
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	* Let WordPress manage the document title.
	* By adding theme support, we declare that this theme does not use a
	* hard-coded <title> tag in the document head, and expect WordPress to
	* provide it for us.
	*/
	add_theme_support( 'title-tag' );

	/*
	* Enable support for Post Thumbnails on posts and pages.
	* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	*/
	add_theme_support( 'post-thumbnails' );

	/*
	* Switch default core markup for search form, comment form, and comments
	* to output valid HTML5.
	*/
	add_theme_support( 'html5', array(
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/*
	* Enable support for Post Formats.
	* See: https://codex.wordpress.org/Post_Formats
	*/
	add_theme_support( 'post-formats', array(
		'aside',
		'image',
		'video',
		'quote',
		'link',
		'gallery',
		'audio',
	) );

	// allow logo to be managed via the WordPress theme customizer
	add_theme_support( 'custom-logo' );



	/**
	* Add support for Gutenberg.
	*
	* @link https://wordpress.org/gutenberg/handbook/reference/theme-support/
	*/
    // Theme supports wide images, galleries and videos.
    add_theme_support( 'align-wide' );
    // Make specific theme colors available in the editor.
    add_theme_support( 'editor-color-palette', array(
			array(
				'name' => 'dark blue',
				'color' => '#1767ef',
			),
			array(
				'name' => 'light gray',
				'color' => '#eee',
			),
			array(
				'name' => 'dark gray',
				'color' => '#444',
			)
		)
    );
}
add_action( 'after_setup_theme', 'dhwp_setup' );


/**
* Enqueue scripts and styles.
*/
function dhwp_scripts() {
	// Theme stylesheets
	///wp_enqueue_style( 'NAME', get_template_directory_uri() . '/style.css' , array( 'DEPENDENCIES' ), '1.0.0 VERSION' ); //this alternative would force the include, rather than allowing the child theme to overwrite it; doesn't seem like we'd every want to disable the ability for the child theme to overwrite

	wp_enqueue_style( 'normalize7', get_theme_file_uri( 'assets/css/normalize7.css' ) , array(), '1.0.0' ); //can be overwritten
	wp_enqueue_style( 'dhwp-base',  get_theme_file_uri( 'assets/css/dhwp-base.css' ) , array( 'normalize7' ), '1.0.0' );
	wp_enqueue_style( 'dhwp-cols-float',  get_theme_file_uri( 'assets/css/dhwp-cols-float.css' ) , array( 'dhwp-base' ), '1.0.0' );
	wp_enqueue_style( 'dhwp-cols-flex',  get_theme_file_uri( 'assets/css/dhwp-cols-flex.css' ) , array( 'dhwp-base' ), '1.0.0' );
	wp_enqueue_style( 'dhwp-cols-grid',  get_theme_file_uri( 'assets/css/dhwp-cols-grid.css' ) , array( 'dhwp-base' ), '1.0.0' );
	wp_enqueue_script( 'jquery-match-height', get_theme_file_uri( 'assets/js/jquery-match-height-min.js' ), array( 'jquery' ), '1.0.0', true );


	// WordPress recommends the following to be included by each theme
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'dhwp_scripts' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function dhwp_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">' . "\n", get_bloginfo( 'pingback_url' ) );
	}
}
add_action( 'wp_head', 'dhwp_pingback_header' );

/**
 * Automated Customizer Setup
 * See functions.php in the child theme to set dhwp_get_customizer_config();
 * @param mixed $wp_customize
 */
function dhwp_customizer_register( $wp_customize ) {
	$priority = 30;
	foreach(dhwp_get_customizer_config() as $section=>$controls){

		//add customizer section
		$priority++;
		$section_slug = sanitize_title_with_dashes($section);
		$wp_customize->add_section( $section_slug , array(
			'title'      => __($section),
			'priority'   => $priority,
		) );

		foreach($controls as $control=>$details){

			//either class or type must be included in $details
			if(array_key_exists('class',$details)||array_key_exists('type',$details)){

				$control_slug = sanitize_title_with_dashes($control);

				//add setting
				$wp_customize->add_setting($control_slug, array(
					'default'   => array_key_exists('default',$details) ? $details['default'] : '',
				));

				//check for and assign className if that's assigned
				$className = false;
				if(array_key_exists('class',$details)) $className = $details['class'];

				//these attributes are removed before being passed to the ->add_control method
				$remove_array = array('css_frontend','css_admin','class','default');
				foreach($remove_array as $remove_detail) unset($details[$remove_detail]);

				$details['section'] = $section_slug;
				$details['label'] = $control;

				if($className){
					//add control for WP_Customize_Color_Control, WP_Customize_Image_Control & WP_Customize_Cropped_Image_Control
					$wp_customize->add_control( new $className( $wp_customize, $control_slug, $details));
				} else {
					//add control
					$wp_customize->add_control($control_slug, $details);
				}

			}
		}
	}
}
add_action( 'customize_register', 'dhwp_customizer_register' );

/**
 * Automated Customizer Setup
 * See functions.php in the child theme to set dhwp_get_customizer_config();
 * @param mixed $array_key
 * @return string
 */
function dhwp_get_customizer_css($array_key) {
	$css = '';

	ob_start();

	foreach(dhwp_get_customizer_config() as $section=>$controls){
		foreach($controls as $control=>$details){
			if(array_key_exists($array_key,$details)){
				$control_slug = sanitize_title_with_dashes($control);
				$value = get_theme_mod( $control_slug, '' );
				if ( ! empty( $value ) ) {
					//cropped images return attachment IDs, so requires extra code to get value
					if(array_key_exists('class',$details) && $details['class']=='WP_Customize_Cropped_Image_Control'){
						$vals = wp_get_attachment_image_src(absint($value),'cropped_image_setting');
						$value = $vals[0]; //this stores the actual URL
					}
					echo str_replace('$$$$',$value,$details[$array_key]);
				}
			}
		}
	}
	$css = ob_get_clean();

    return $css;
}
function dhwp_get_customizer_css_frontend() {
	return dhwp_get_customizer_css('css_frontend');
}
function dhwp_get_customizer_css_admin() {
	return dhwp_get_customizer_css('css_admin');
}



// DIGITAL HILL GLOBAL FUNCTIONS

/**
 * Get all pages that are assigned a certain template
 * @param mixed $template 
 * @param mixed $args 
 * @return array|boolean
 */
function dh_get_pages_by_template( $template = '', $args = array() ) {
	if ( empty($template) ) return false;
	if ( strpos($template, '.php') !== (strlen($template) - 4) )  $template .= '.php';
	$args['meta_key'] = '_wp_page_template';
	$args['meta_value'] = $template;
	return get_pages($args);
}

if(!function_exists('dh_print_r')){
	/**
	 * Wraps <pre> around built-in print_r() function
	 * @param mixed $a 
	 * @param mixed $return 
	 * @return boolean|string
	 */
	function dh_print_r($a, $return = false) {
		$string = '<pre>' . print_r($a, true) . '</pre>';
		if($return){
			return $string;
		} else {
			echo $string;
			return true;
		}
	}
}

/**
 * Geolocate using geo.digitalhill.com
 * @param mixed $location The address to geolocate
 * @param mixed $key The key for access to geo.digitalhill.com
 * @return mixed
 */
function dh_get_geo($location,$key){
	$ch = curl_init();
	curl_setopt($ch, CURLOPT_URL, 'https://geo.digitalhill.com/?address='.urlencode($location).'&key='.$key);
	curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
	$result = curl_exec($ch);
	curl_close($ch);
	return json_decode($result,true);
}

/**
 * Find the distance between two sets of latitude and longitudes
 * @param mixed $lat1 
 * @param mixed $lon1 
 * @param mixed $lat2 
 * @param mixed $lon2 
 * @return float|integer
 */
function dh_get_distance($lat1, $lon1, $lat2, $lon2) {
	$theta = $lon1 - $lon2;
	$dist = sin(deg2rad($lat1)) * sin(deg2rad($lat2)) +  cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * cos(deg2rad($theta));
	$dist = acos($dist);
	$dist = rad2deg($dist);
	$miles = $dist * 60 * 1.1515;
	return $miles;
}

/**
 * Move Yoast to the bottom of the editor (mostly to keep ACF fields above it)
 * @return string
 */
function yoasttobottom() {
	return 'low';
}
add_filter( 'wpseo_metabox_prio', 'yoasttobottom');

// Plugin Update Checker Release 4.4
// See: https://github.com/YahnisElsts/plugin-update-checker/
if(!class_exists('Puc_v4_Factory',false)) require_once get_template_directory().'/includes/plugin-update-checker/plugin-update-checker.php';
$dhwpBaseChecker = Puc_v4_Factory::buildUpdateChecker(
	'http://plugins.digitalhill.com/dhthemes/dhwp-base/update.json',
	__FILE__, //Full path to the main plugin file or functions.php.
	'dhwp-base'
);
